# Unfree packages

PacketMix has no moral objection to you installing unfree packages on your own system. There are, however, legal restrictions to caching unfree packages from CI builds.

The solution is to use PacketMix's `lib.ci` to test whether you're running in CI and only install the unfree package when you are not.

> [!WARN]
> Using this technique will force you to build your own unfree packages on system upgrades. This may not be a problem for packages like `discord` (which are mostly just a download anyway) but for source-available packages with significant compilation like `mongodb` or `stalwart-mail-enterprise` this may take a lot of build time at inopportune moments. 
> 
> Consider freezing inputs, running additional private binary caches, scheduling updates for points where you expect to have significant resources free or finding alternatives to mitigate this.

## Installing unfree packages directly

Let's say you want to install `hello` and `hello-unfree` like so:

```nix
# This code will not pass CI, since as it unconditionally installs an unfree package
{ pkgs, ... }:
{
  environment.systemPackages = [
    pkgs.hello
    pkgs.hello-unfree
  ];
}
```

In ingredients, you can access the main nilla project by adding the `project` argument to your module.

```diff
  # This code will not pass CI, since as it unconditionally installs an unfree package
- { pkgs, ... }:
+ { project, pkgs, ... }:
  {
    environment.systemPackages = [
      pkgs.hello
      pkgs.hello-unfree
    ];
  }
```

You can use the `ci` attribute of `project.lib` to test if you're in a CI environment, and only install `hello-unfree` if you're not in CI.

```nix
{ project, pkgs, ... }:
{
  environment.systemPackages = [
    pkgs.hello
  ]
  ++ (
    if project.lib.ci then
      [ ]
    else
      [
        pkgs.hello-unfree
      ]
  );
}
```

## Installing with a module

### With a viable free alternative

Some unfree packages may have viable free alternatives. This is particularly common with enterprise editions of software. For example, say you are installing Stalwart Mail's enterprise edition, `pkgs.stalwart-mail-enterprise`, this could be replaced with `pkgs.stalwart-mail` as a drop-in with reduced functionality.

```
# This code will not pass CI, since as it unconditionally installs an unfree package
{ pkgs, ... }:
{
  services.stalwart-mail = {
    enable = true;
    package = pkgs.stalwart-mail-enterprise;
  };
}
```

As in the directly-installed example, we can use `project.lib.ci` to determine whether to install the (unfree) enterprise edition or the (free) community edition.

```
{ project, pkgs, ... }:
{
  services.stalwart-mail = {
    enable = true;
      package =
        if project.lib.ci then
          project.inputs.nixos-unstable.result.x86_64-linux.stalwart-mail
        else
          project.inputs.nixos-unstable.result.x86_64-linux.stalwart-mail-enterprise;
  };
}
```

### With no viable free alternative

Some modules may install unfree packages without a viable alternative. For example, there are no longer any free versions of `mongodb` in nixpkgs.

```
# This code will not pass CI, since as it unconditionally installs an unfree package
{
  services.mongodb.enable = true;
}
```

In this case, you should conditionally enable the module based on `project.lib.ci`.

```
# This code will not pass CI, since as it unconditionally installs an unfree package
{ project, ... }:
{
  services.mongodb.enable = !project.lib.ci;
}
```